/*global define */
/*jshint esversion: 6 */

/*
	Container:

	This object provides a mixin for each node container.
*/

define ([   "src/build/Node",   "src/math/Mat3",			"src/math/Vec2",
			"src/utils",		"src/build/treeContainer",	"lodash"],
function(	Node,				mat3,						vec2,
			utils,				treeContainer,				lodash
		) {
	"use strict";

	var WarpDomain = {
			BOX    : "box",
			CONTOUR: "contour"
		},
		Node_clone = Node.prototype.clone;

	function Warper () {
		this.restMatParent_Puppet = null;
		this.aRestMatPuppet_Handle = [];
		this.aRestMatHandle_Puppet = [];
		this.aLocations = null;
		this.aAnchors = null;
		this.warpDomain = WarpDomain.BOX;
		this.warpMeshExpansion = 0;
		this.warpMaxNumTrianglesHint = -1;
		this.warpMinTriangleAreaHint = 0;
		this.canWarp = false;
	}

	utils.mixin(Warper, 
		// Keep in sync with Stage.cpp
		{
			canInitialize : function () {
				if (this.restMatParent_Puppet) {
					return this.aRestMatPuppet_Handle.length > 1;
				}

				return false;
			},

			getRestLocations : function () {
				var aRestMatPuppet_Handle = this.aRestMatPuppet_Handle;
				if (aRestMatPuppet_Handle) {
					// no modification allowed
					return lodash.map(this.aRestMatPuppet_Handle, function (mi) {
						return mat3.getTranslation(mi);
					});
				}
				// no handle locations available: forgot call to initWarp()?
				utils.assert(false, "getRestLocations(): missing rest configuration.");
			}
		});

	function Container(inName) {
		this.Node("Container", inName);
		this.m_children = [];

		this.privateWarper = new Warper();
	}

	Container.WarpDomain = WarpDomain;

	utils.mixin(Container, 
		Node,
		treeContainer({ 
			childrenLabel   : "m_children"
		}),
		{
			exclusivelyEnableChild :function (inChildName) {
				var c, foundNode, nodesToDisable = [];

				for (c = 0; c < this.m_children.length; c += 1) {
					if (this.m_children[c].getName() === inChildName) {
						foundNode = this.m_children[c];
					} else if (this.m_children[c].getVisibleEnabled()) {
						nodesToDisable.push(this.m_children[c]);
					}
				}

				if (foundNode && !foundNode.getVisibleEnabled()) {
					foundNode.setVisible(true);
					for (c = 0; c < nodesToDisable.length; c += 1) {
						nodesToDisable[c].setVisible(false);
					}
				}
			},

			clone : function (clone_children, other) {
				var i, ci,
					result = other;

				if (result) {
					// init
					Container.call(result);
				} else {
					// alloc and init
					result = new Container();
				}

				// clone node state
				Node_clone.call(this, clone_children, result);

				result.privateWarper = lodash.cloneDeep(this.privateWarper);

				// copy (by default) container children...
				if (clone_children === undefined || clone_children) {
					result.m_children = [];
					for (i = 0; i < this.m_children.length; i += 1) {
						ci = this.m_children[i];
						ci = ci.clone(clone_children);
						result.m_children[i] = ci;
						ci.setParent(result);
					}
				}

				return result;
			},

			setEnabledChildren : function (enabled) {
				var c;
				for (c = 0; c < this.m_children.length; c += 1) {
					this.m_children[c].setEnabled(enabled);
				}
			},
		
			removeChangeNotifiers: function () {
				var c;
				for (c = 0; c < this.m_children.length; c += 1) {
					this.m_children[c].removeChangeNotifiers();
				}
				this.removeAllChangeNotifiers();
			},

			// Warper
			setWarpDomain : function (type) {
				this.privateWarper.warpDomain = type;
			},

			getWarpDomain : function () {
				return this.privateWarper.warpDomain;
			},

			setWarpMeshExpansion : function (expansion) {
				this.privateWarper.warpMeshExpansion = expansion;
			},

			getWarpMeshExpansion : function () {
				return this.privateWarper.warpMeshExpansion;
			},
			
			setWarpMaxNumTrianglesHint : function (warpMaxNumTrianglesHint) {
				this.privateWarper.warpMaxNumTrianglesHint = warpMaxNumTrianglesHint;
			},

			getWarpMaxNumTrianglesHint : function () {
				return this.privateWarper.warpMaxNumTrianglesHint;
			},
			
			setWarpMinTriangleAreaHint : function (warpMinTriangleAreaHint) {
				this.privateWarper.warpMinTriangleAreaHint = warpMinTriangleAreaHint;
			},

			getWarpMinTriangleAreaHint : function () {
				return this.privateWarper.warpMinTriangleAreaHint;
			},
							
			setWarpHash : function (hash) {
				this.privateWarper.warpHash = hash;
			},

			getWarpHash : function () {
				return this.privateWarper.warpHash;
			},

			setWarpRest : function (aMatPuppet_HandleInitial, aLocations, aAnchors) {
				// store initial/rest state...
				this.privateWarper.aRestMatPuppet_Handle = aMatPuppet_HandleInitial;
				this.privateWarper.restMatParent_Puppet = this.getMatrix();
				this.privateWarper.aRestMatHandle_Puppet = lodash.map(this.privateWarper.aRestMatPuppet_Handle, function (m) { 
					return mat3.invert(m);
				});

				this.privateWarper.aLocations = aLocations;
				this.privateWarper.aAnchors = aAnchors;
			},

			canWarp : function () {
				return this.privateWarper.canWarp;
			},

			shouldInitializeWarp : function () {
				return this.privateWarper.canInitialize();
			},

			/** 
			 * Warp container subtree.
			 * @param aMatContainer_Handle Array of affine transforms relative to local/container frame.
			 * @param aAutomationAttribute Array of logically ORed attributes indicating degrees of freedom in each transform.
			 * @return Boolean indicating whether warp was applied (true) or short-circuited (false) by updating container transform instead.
			 */
			warp : function (aMatContainer_Handle, aAutomationAttribute) {
				var warper = this.privateWarper;

				utils.assert(aMatContainer_Handle.length === aAutomationAttribute.length, "warp(): inconsistent arguments");

				if ( !warper.aRestMatPuppet_Handle ) return;

				if (warper.canWarp) {
					warper.cpp.warp(aMatContainer_Handle, aAutomationAttribute);
					this.callChangeNotifiers("warpChanged");
				}
			},

			calcPointCoord : function (inPoints) {
				var native = this.privateWarper.cpp,
					points = [],
					ids = [],
					lambdas = [];

				native.calcPointCoord(inPoints, points, ids, lambdas);
				return { points, ids, lambdas };
			},

			warpPoint : function (aMatContainer_Handle, coords) {
				var native = this.privateWarper.cpp,
					warps = [];

				native.warpPoint(aMatContainer_Handle, coords.points, coords.ids, coords.lambdas, warps);

				return warps;
			},

			/**
			 * Called by Zoot's native stage.
			 * @param warper Component in charge of warper computation.
			 */
			addWarperComponent : function (native) {
				this.privateWarper.cpp = native;

				// native implementation should contain these methods
				utils.assert(native.init, "addWarperComponent: expected object with 'init' function.");
				utils.assert(native.warp, "addWarperComponent: expected object with 'warp' function.");
				utils.assert(native.warpGeometry, "addWarperComponent: expected object with 'warpGeometry' function.");
				/* 	function calcPointCoords(points : AofA<Number>, ids : Array<Number>, coords : AofA<Number>) */
				utils.assert(native.calcPointCoord, "addWarperComponent: expected object with 'calcPointCoord' function.");
				/* 	function warpPoint(dofs : AofA<Number>, points : AofA<Number>, ids : Array<Number>, coords : AofA<Number>, warps : AofA<Number>) */
				utils.assert(native.warpPoint, "addWarperComponent: expected object with 'warpPoint' function.");
				/* 	function getMeshOutline(result : AofA<Number>) */
				utils.assert(native.getMeshOutline, "addWarperComponent: expected object with 'getMeshOutline' function.");
				/* 	function getMeshGeometry(result : AofA<Number>) */
				utils.assert(native.getMeshGeometry, "addWarperComponent: expected object with 'getMeshGeometry' function.");
				/* 	function getMeshTopology(result : AofA<Number>) */
				utils.assert(native.getMeshTopology, "addWarperComponent: expected object with 'getMeshTopology' function.");
				/* 	function getMeshWeights(result : AofA<Number>) */
				utils.assert(native.getMeshWeights, "addWarperComponent: expected object with 'getMeshWeights' function.");
				/* 	function getWarpGeometry(result : AofA<Number>) */
				utils.assert(native.getWarpGeometry, "addWarperComponent: expected object with 'getWarpGeometry' function.");
				/* 	function getOutlineWarpGeometry(result : AofA<Number>) */
				utils.assert(native.getOutlineWarpGeometry, "addWarperComponent: expected object with 'getOutlineWarpGeometry' function.");
			},

			getOutlineWarpGeometry : function () {
				var warper = this.privateWarper,
					native = warper.cpp,
					outline = [];

				if (!warper || !warper.canWarp) return outline;

				native.getOutlineWarpGeometry(outline);

				return outline;
			},

			getMeshOutline : function () {
				var warper = this.privateWarper,
					native = warper.cpp,
					outline = [];

				if (!warper || !warper.canWarp) return outline;

				native.getMeshOutline(outline);

				if (outline.length > 0) {
				} else {
					// construct outline from bounding box
					var ltwh = this.getMeshBounds();
					// with desugaring semantics:
					// let [l, t, w, h] = this.getMeshBounds();

					outline[0] = [ ltwh[0], ltwh[1] ];
					outline[1] = [ ltwh[0], ltwh[1] + ltwh[3]];
					outline[2] = [ ltwh[0] + ltwh[2], ltwh[1] + ltwh[3] ];
					outline[3] = [ ltwh[0] + ltwh[2], ltwh[1] ];
				}

				return outline;
			},

			initWarp : function () {
				var warper = this.privateWarper;
				warper.canWarp = warper.cpp.init(warper.warpHash, warper.aLocations, warper.aAnchors, warper.warpDomain, warper.warpMeshExpansion, warper.warpMaxNumTrianglesHint, warper.warpMinTriangleAreaHint);
			
			}

		});


	return Container;
});
